// Utility functions for Indian Standard Time (IST) formatting

/**
 * Get current date and time in Indian Standard Time
 * @returns {Date} Current date in IST
 */
export const getCurrentIST = () => {
  const now = new Date();
  // IST is UTC+5:30
  const istOffset = 5.5 * 60 * 60 * 1000; // 5.5 hours in milliseconds
  const istTime = new Date(now.getTime() + istOffset);
  return istTime;
};

/**
 * Format date to IST string
 * @param {Date|string} date - Date to format
 * @param {string} format - Format type ('date', 'time', 'datetime', 'full')
 * @returns {string} Formatted date string in IST
 */
export const formatIST = (date, format = "datetime") => {
  if (!date) return "N/A";

  const dateObj = typeof date === "string" ? new Date(date) : date;

  // IST is UTC+5:30
  const istOffset = 5.5 * 60 * 60 * 1000;
  const istTime = new Date(dateObj.getTime() + istOffset);

  // Helper function to format date as DD-MM-YYYY
  const formatDateAsDDMMYYYY = (date) => {
    const day = date.getDate().toString().padStart(2, '0');
    const month = (date.getMonth() + 1).toString().padStart(2, '0');
    const year = date.getFullYear();
    return `${day}-${month}-${year}`;
  };

  // Helper function to format time as HH:MM:SS AM/PM
  const formatTimeAsHHMMSS = (date) => {
    return date.toLocaleTimeString("en-IN", {
      timeZone: "Asia/Kolkata",
      hour: "2-digit",
      minute: "2-digit",
      second: "2-digit",
      hour12: true,
    });
  };

  switch (format) {
    case "date":
      return formatDateAsDDMMYYYY(istTime);
    case "time":
      return formatTimeAsHHMMSS(istTime);
    case "datetime":
      return `${formatDateAsDDMMYYYY(istTime)} ${formatTimeAsHHMMSS(istTime)}`;
    case "full":
      return `${formatDateAsDDMMYYYY(istTime)} ${formatTimeAsHHMMSS(istTime)}`;
    default:
      return `${formatDateAsDDMMYYYY(istTime)} ${formatTimeAsHHMMSS(istTime)}`;
  }
};

/**
 * Get current IST time string
 * @returns {string} Current time in IST format
 */
export const getCurrentISTTime = () => {
  return formatIST(new Date(), "time");
};

/**
 * Get current IST date string
 * @returns {string} Current date in IST format
 */
export const getCurrentISTDate = () => {
  return formatIST(new Date(), "date");
};

/**
 * Get current IST datetime string
 * @returns {string} Current datetime in IST format
 */
export const getCurrentISTDateTime = () => {
  return formatIST(new Date(), "datetime");
};

/**
 * Format timestamp for display with IST
 * @param {string|Date} timestamp - Timestamp to format
 * @returns {string} Formatted timestamp in IST
 */
export const formatTimestamp = (timestamp) => {
  if (!timestamp) return "N/A";

  const date = new Date(timestamp);
  const now = new Date();
  const diffInHours = (now - date) / (1000 * 60 * 60);

  if (diffInHours < 24) {
    // Less than 24 hours ago - show time
    return formatIST(date, "time");
  } else if (diffInHours < 168) {
    // Less than a week ago - show day and time
    return formatIST(date, "datetime");
  } else {
    // More than a week ago - show full date
    return formatIST(date, "date");
  }
};

/**
 * Get relative time in IST (e.g., "2 hours ago", "3 days ago")
 * @param {string|Date} timestamp - Timestamp to get relative time for
 * @returns {string} Relative time string
 */
export const getRelativeTime = (timestamp) => {
  if (!timestamp) return "N/A";

  const date = new Date(timestamp);
  const now = new Date();
  const diffInSeconds = Math.floor((now - date) / 1000);

  if (diffInSeconds < 60) {
    return "Just now";
  } else if (diffInSeconds < 3600) {
    const minutes = Math.floor(diffInSeconds / 60);
    return `${minutes} minute${minutes > 1 ? "s" : ""} ago`;
  } else if (diffInSeconds < 86400) {
    const hours = Math.floor(diffInSeconds / 3600);
    return `${hours} hour${hours > 1 ? "s" : ""} ago`;
  } else if (diffInSeconds < 2592000) {
    const days = Math.floor(diffInSeconds / 86400);
    return `${days} day${days > 1 ? "s" : ""} ago`;
  } else {
    return formatIST(date, "date");
  }
};

/**
 * Format date to Indian standard format (DD-MM-YYYY)
 * @param {string|Date} date - Date to format
 * @returns {string} Date in DD-MM-YYYY format
 */
export const formatDateIndian = (date) => {
  if (!date) return "N/A";
  
  const dateObj = typeof date === "string" ? new Date(date) : date;
  
  // Format as DD-MM-YYYY
  const day = dateObj.getDate().toString().padStart(2, '0');
  const month = (dateObj.getMonth() + 1).toString().padStart(2, '0');
  const year = dateObj.getFullYear();
  
  return `${day}-${month}-${year}`;
};
