import React from 'react';
import { useTheme } from '../../contexts/ThemeContext';
import { FaUsers, FaSearch, FaDownload } from 'react-icons/fa';

const EmployeeStats = ({ totalEmployees, searchTerm, setSearchTerm, exportToExcel }) => {
  const { isLight } = useTheme();

  const cardBgClass = isLight 
    ? 'bg-white border border-yellow-200 shadow-md hover:shadow-lg' 
    : 'bg-gradient-to-br from-gray-800 to-slate-800 border border-gray-700 shadow-lg hover:shadow-xl';
  
  const headingClass = isLight 
    ? 'text-amber-800' 
    : 'text-gray-100';
  
  const textClass = isLight 
    ? 'text-amber-700' 
    : 'text-gray-300';

  return (
    <>
      <div className={`${cardBgClass} rounded-xl px-4 sm:px-6 py-3 sm:py-4 transition-all duration-300`}>
        <div className="flex flex-col lg:flex-row lg:items-center justify-between gap-4 lg:gap-6 min-w-0">
          {/* Total Employees - Modified for mobile layout */}
          <div className="flex items-center justify-between lg:justify-start gap-3 min-w-0 flex-shrink-0 w-full lg:w-auto">
            {/* Left side - Heading and icon */}
            <div className="flex items-center gap-3 min-w-0">
              <div className="text-blue-500 text-xl sm:text-2xl">
                <FaUsers />
              </div>
              <div className="min-w-0 text-left">
                <p className={`${textClass} text-xs sm:text-sm uppercase tracking-wide`}>Total Employees</p>
                <p className={`${textClass} text-xs sm:text-sm`}>Active in system</p>
              </div>
            </div>
            
            {/* Right side - Employee number (mobile) */}
            <div className="lg:hidden text-right">
              <p className={`text-xl sm:text-2xl font-bold ${headingClass}`}>{totalEmployees}</p>
            </div>
            
            {/* Desktop view - Employee number below heading */}
            <div className="hidden lg:block min-w-0">
              <p className={`text-xl sm:text-2xl font-bold ${headingClass}`}>{totalEmployees}</p>
            </div>
          </div>
          
          {/* Search and Export */}
          <div className="flex flex-col sm:flex-row items-stretch sm:items-center gap-3 min-w-0 w-full lg:w-auto">
            {/* Search */}
            <div className="relative w-full sm:flex-1 sm:max-w-xs lg:max-w-sm">
              <FaSearch className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 text-sm" />
              
              <input
                type="text"
                placeholder="Search employees..."
                value={searchTerm}
                onChange={(e) => setSearchTerm(e.target.value)}
                className={`w-full pl-10 pr-4 py-2 rounded-full transition-all duration-300 text-left outline-none border text-sm ${
                  isLight 
                    ? 'bg-white text-gray-900 placeholder-gray-500 border-gray-300' 
                    : 'bg-gray-700 text-gray-100 placeholder-gray-400 border-gray-600'
                }`}
              />
            </div>

            {/* Export Button - Hidden on mobile */}
            <div className="relative group flex-shrink-0 self-center hidden md:block">
              <button
                onClick={exportToExcel}
                className={`w-8 h-8 rounded-full flex items-center justify-center transition-all duration-300 hover:scale-110 ${
                  isLight
                    ? 'bg-green-500 hover:bg-green-600 text-white'
                    : 'bg-green-600 hover:bg-green-700 text-white'
                }`}
                title="Download Excel File"
              >
                <FaDownload className="text-xs" />
              </button>
              
              {/* Responsive Tooltip */}
              <div className={`absolute bottom-full left-1/2 transform -translate-x-1/2 mb-2 px-3 py-2 rounded-lg text-xs font-medium opacity-0 group-hover:opacity-100 transition-all duration-300 pointer-events-none z-30 ${
                isLight
                  ? 'bg-gray-900 text-white shadow-xl'
                  : 'bg-white text-gray-900 shadow-xl border border-gray-300'
              } whitespace-nowrap`}>
                Download Excel
                {/* Tooltip Arrow */}
                <div className={`absolute top-full left-1/2 transform -translate-x-1/2 w-0 h-0 border-l-4 border-r-4 border-t-4 border-transparent ${
                  isLight
                    ? 'border-t-gray-900'
                    : 'border-t-white'
                }`}></div>
              </div>
            </div>
          </div>
        </div>
      </div>
    </>
  );
};

export default EmployeeStats; 