import React from 'react';
import { useTheme } from '../../contexts/ThemeContext';
import { FaChevronLeft, FaChevronRight } from 'react-icons/fa';

const ActivityLogTable = ({ 
  employeeLogData, 
  loading, 
  error,
  currentPage, 
  totalPages, 
  startIndex, 
  endIndex, 
  goToPage, 
  goToPrevious, 
  goToNext 
}) => {
  const { isLight } = useTheme();

  // Helper function to format project names for display
  const formatProjectNames = (projectNames) => {
    if (!projectNames || projectNames === 'no_projects') {
      return 'No Projects';
    }
    return projectNames;
  };

  const cardBgClass = isLight 
    ? 'bg-white border border-yellow-200 shadow-md hover:shadow-lg' 
    : 'bg-slate-800 border border-slate-700 shadow-xl hover:shadow-2xl';
  
  const headingClass = isLight 
    ? 'text-amber-800' 
    : 'text-slate-100';
  
  const textClass = isLight 
    ? 'text-amber-700' 
    : 'text-slate-300';



  return (
    <div className={`${cardBgClass} rounded-xl overflow-hidden transition-all duration-300`}>
      <div className="overflow-x-auto">
        <table className="w-full table-fixed min-w-[800px]">
          <thead className={`${isLight ? 'bg-gradient-to-r from-yellow-200 to-amber-200' : 'bg-slate-700'}`}>
            <tr>
              <th className={`px-2 sm:px-3 py-2 text-center font-semibold text-xs ${headingClass}`} style={{ width: '8%' }}>Log ID</th>
              <th className={`px-2 sm:px-3 py-2 text-center font-semibold text-xs ${headingClass}`} style={{ width: '8%' }}>User ID</th>
              <th className={`px-2 sm:px-3 py-2 text-center font-semibold text-xs ${headingClass}`} style={{ width: '15%' }}>Employee</th>
              <th className={`px-2 sm:px-3 py-2 text-center font-semibold text-xs ${headingClass}`} style={{ width: '12%' }}>Division</th>
              <th className={`px-2 sm:px-3 py-2 text-center font-semibold text-xs ${headingClass}`} style={{ width: '12%' }}>Role</th>
              <th className={`px-2 sm:px-3 py-2 text-center font-semibold text-xs ${headingClass}`} style={{ width: '15%' }}>Projects</th>
              <th className={`px-2 sm:px-3 py-2 text-center font-semibold text-xs ${headingClass}`} style={{ width: '10%' }}>Action</th>
              <th className={`px-2 sm:px-3 py-2 text-center font-semibold text-xs ${headingClass}`} style={{ width: '12%' }}>Timestamp</th>
              <th className={`px-2 sm:px-3 py-2 text-center font-semibold text-xs ${headingClass}`} style={{ width: '8%' }}>Remarks</th>
            </tr>
          </thead>
          <tbody className={isLight ? 'bg-white' : 'bg-gray-800'}>
            {loading ? (
              <tr>
                <td colSpan="9" className="p-4 sm:p-8 text-center">
                  <div className={`${textClass}`}>
                    <div className="animate-spin h-6 w-6 sm:h-8 sm:w-8 border-4 border-blue-600 border-t-transparent rounded-full mx-auto mb-4"></div>
                    <p className="text-base sm:text-lg font-medium">Loading employee logs...</p>
                  </div>
                </td>
              </tr>
            ) : error ? (
              <tr>
                <td colSpan="9" className="p-4 sm:p-8 text-center">
                  <div className={`${textClass}`}>
                    <div className="text-red-500 text-3xl sm:text-4xl mb-4">⚠️</div>
                    <p className="text-base sm:text-lg font-medium mb-2 text-red-500">Error Loading Data</p>
                    <p className="text-sm opacity-75">{error}</p>
                  </div>
                </td>
              </tr>
            ) : employeeLogData && employeeLogData.length > 0 ? (
              employeeLogData.map((log, index) => (
                <tr 
                  key={log.id} 
                  className={`border-b transition-colors duration-200 ${
                    isLight ? 'border-gray-200 hover:bg-yellow-50' : 'border-slate-600 hover:bg-slate-700'
                  } ${
                    index % 2 === 0 
                      ? (isLight ? 'bg-gray-50' : 'bg-gray-750') 
                      : (isLight ? 'bg-white' : 'bg-gray-800')
                  }`}
                >
                  <td className={`px-1 sm:px-2 py-2 text-center font-mono text-xs font-medium ${isLight ? 'text-blue-600' : 'text-cyan-400'}`}>
                    #{log.id}
                  </td>
                  <td className={`px-1 sm:px-2 py-2 text-center font-mono text-xs ${isLight ? 'text-gray-600' : 'text-slate-400'}`}>{log.user_id}</td>
                  <td className={`px-1 sm:px-2 py-2 text-center font-medium text-xs truncate ${textClass}`} title={log.employee}>{log.employee}</td>
                  <td className={`px-1 sm:px-2 py-2 text-center text-xs truncate ${textClass}`} title={log.division || 'N/A'}>{log.division || 'N/A'}</td>
                  <td className={`px-1 sm:px-2 py-2 text-center text-xs truncate ${textClass}`} title={log.role || 'N/A'}>{log.role || 'N/A'}</td>
                  <td className={`px-1 sm:px-2 py-2 text-center text-xs ${textClass}`}>
                    <div className="truncate" title={formatProjectNames(log.projects)}>
                      {formatProjectNames(log.projects)}
                    </div>
                  </td>
                  <td className={`px-1 sm:px-2 py-2 text-center text-xs truncate ${textClass}`} title={log.action}>{log.action}</td>
                  <td className={`px-1 sm:px-2 py-2 text-center text-xs ${isLight ? 'text-gray-600' : 'text-slate-400'}`}>{log.timestamp}</td>
                  <td className={`px-1 sm:px-2 py-2 text-center text-xs ${isLight ? 'text-gray-600' : 'text-slate-400'}`}>
                    <div className="truncate" title={log.remarks}>
                      {log.remarks || 'N/A'}
                    </div>
                  </td>
                </tr>
              ))
            ) : (
              <tr>
                <td colSpan="9" className="p-4 sm:p-8 text-center">
                  <div className={`${textClass}`}>
                    <div className="text-3xl sm:text-4xl mb-4">📋</div>
                    <p className="text-base sm:text-lg font-medium mb-2">No activity logs found</p>
                    <p className="text-sm opacity-75">Try adjusting your date range or check back later.</p>
                  </div>
                </td>
              </tr>
            )}
          </tbody>
        </table>
      </div>

      {/* Pagination Controls - Matching Employees section styling */}
      {totalPages > 1 && (
        <div className={`mt-0 px-3 py-2 border-t overflow-hidden ${
          isLight 
            ? 'bg-gray-50 border-gray-200' 
            : 'bg-gray-700 border-gray-600'
        }`}>
          <div className="flex flex-col sm:flex-row justify-between items-center gap-3 sm:gap-2 min-w-0">
            {/* Page Info */}
            <div className={`text-xs flex-shrink-0 text-center sm:text-left ${
              isLight ? 'text-gray-600' : 'text-gray-300'
            }`}>
              {startIndex + 1}-{Math.min(endIndex, employeeLogData.length)} of {employeeLogData.length}
            </div>

            {/* Pagination Controls */}
            <div className="flex items-center justify-center gap-2">
              {/* Previous Button */}
              <button
                onClick={goToPrevious}
                disabled={currentPage === 1}
                className={`flex items-center justify-center gap-1 px-3 py-2 rounded-lg text-sm ${
                  currentPage === 1
                    ? (isLight ? 'bg-gray-100 text-gray-400 cursor-not-allowed' : 'bg-gray-600 text-gray-500 cursor-not-allowed')
                    : (isLight ? 'bg-white hover:bg-gray-100 text-gray-700 shadow-sm' : 'bg-gray-600 hover:bg-gray-500 text-gray-200')
                }`}
              >
                <FaChevronLeft className="text-xs flex-shrink-0" />
                <span>Prev</span>
              </button>

              {/* Current Page */}
              <div className={`px-3 py-2 rounded-lg font-medium text-sm ${
                isLight ? 'bg-amber-500 text-white' : 'bg-gray-300 text-gray-900'
              }`}>
                {currentPage}
              </div>

              {/* Next Button */}
              <button
                onClick={goToNext}
                disabled={currentPage === totalPages}
                className={`flex items-center justify-center gap-1 px-3 py-2 rounded-lg text-sm ${
                  currentPage === totalPages
                    ? (isLight ? 'bg-gray-100 text-gray-400 cursor-not-allowed' : 'bg-gray-600 text-gray-500 cursor-not-allowed')
                    : (isLight ? 'bg-white hover:bg-gray-100 text-gray-700 shadow-sm' : 'bg-gray-600 hover:bg-gray-500 text-gray-200')
                }`}
              >
                <span>Next</span>
                <FaChevronRight className="text-xs flex-shrink-0" />
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default ActivityLogTable; 