import React, { useState, useEffect } from "react";
import { useTheme } from "../../contexts/ThemeContext";
import {
  FaTimes,
  FaUsers,
  FaSpinner,
  FaExclamationTriangle,
  FaUserTie,
  FaBuilding,
} from "react-icons/fa";
import { buildApiUrl, API_ENDPOINTS } from "../../utils/config";

const DepartmentEmployeesModal = ({ isOpen, onClose, department }) => {
  const { isLight } = useTheme();
  const [employees, setEmployees] = useState([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);

  useEffect(() => {
    if (isOpen && department) {
      fetchEmployees();
    }
  }, [isOpen, department]);

  // Prevent body scroll when modal is open
  useEffect(() => {
    if (isOpen) {
      document.body.style.overflow = "hidden";
    } else {
      document.body.style.overflow = "unset";
    }

    return () => {
      document.body.style.overflow = "unset";
    };
  }, [isOpen]);

  const fetchEmployees = async () => {
    try {
      setLoading(true);
      setError(null);

      const response = await fetch(
        buildApiUrl(API_ENDPOINTS.EMPLOYEES_BY_DIVISION_ACCURATE(department.id))
      );

      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const data = await response.json();
      setEmployees(data);
    } catch (err) {
      setError(err.message);
    } finally {
      setLoading(false);
    }
  };

  // Handle backdrop click
  const handleBackdropClick = (e) => {
    if (e.target === e.currentTarget) {
      onClose();
    }
  };

  // Handle escape key
  useEffect(() => {
    const handleEscape = (e) => {
      if (e.key === "Escape" && isOpen) {
        onClose();
      }
    };

    if (isOpen) {
      document.addEventListener("keydown", handleEscape);
    }

    return () => {
      document.removeEventListener("keydown", handleEscape);
    };
  }, [isOpen, onClose]);

  if (!isOpen) return null;

  return (
    <div
      className="fixed inset-0 flex items-center justify-center z-50 p-4"
      style={{ backgroundColor: "rgba(0, 0, 0, 0.1)" }}
      onClick={handleBackdropClick}
    >
      <div
        className="absolute inset-0"
        onClick={onClose}
        style={{ cursor: "pointer" }}
        title="Click to close"
      />
      <div
        className={`${
          isLight ? "bg-white" : "bg-gray-800"
        } rounded-xl shadow-2xl max-w-2xl w-full max-h-[80vh] overflow-hidden transform transition-all duration-200 ease-out relative z-60 border ${
          isLight ? "border-gray-200" : "border-gray-700"
        }`}
        onClick={(e) => e.stopPropagation()}
      >
        {/* Header */}
        <div
          className={`flex items-center justify-between p-6 border-b ${
            isLight ? "border-gray-200" : "border-gray-700"
          }`}
        >
          <div className="flex items-center gap-3">
            <div
              className={`p-2 rounded-lg ${
                isLight ? "bg-blue-100" : "bg-blue-900"
              }`}
            >
              <FaUsers className={`text-blue-600 text-xl`} />
            </div>
            <div>
              <h2
                className={`text-xl font-bold ${
                  isLight ? "text-gray-900" : "text-white"
                }`}
              >
                {department.name} Employees
              </h2>
              <p
                className={`text-sm ${
                  isLight ? "text-gray-600" : "text-gray-400"
                }`}
              >
                Employee List
              </p>
            </div>
          </div>
          <button
            onClick={onClose}
            className={`p-2 rounded-lg transition-colors flex items-center justify-center w-8 h-8 ${
              isLight
                ? "hover:bg-gray-100 text-gray-500 hover:text-gray-700"
                : "hover:bg-gray-700 text-gray-400 hover:text-gray-200"
            }`}
            aria-label="Close modal"
          >
            <FaTimes className="text-xl flex-shrink-0" />
          </button>
        </div>

        {/* Content */}
        <div
          className={`p-6 overflow-y-auto ${
            employees.length > 3 ? "max-h-[50vh]" : "max-h-none"
          }`}
        >
          {loading && (
            <div className="flex items-center justify-center py-8">
              <FaSpinner className="animate-spin text-2xl mr-3 text-blue-600" />
              <span
                className={`text-lg ${
                  isLight ? "text-gray-700" : "text-gray-300"
                }`}
              >
                Loading employees...
              </span>
            </div>
          )}

          {error && (
            <div className="flex flex-col items-center justify-center py-8">
              <FaExclamationTriangle className="text-3xl mb-3 text-red-500" />
              <h3
                className={`text-lg font-semibold mb-2 ${
                  isLight ? "text-red-600" : "text-red-400"
                }`}
              >
                Error Loading Employees
              </h3>
              <p
                className={`text-sm mb-4 text-center ${
                  isLight ? "text-red-500" : "text-red-400"
                }`}
              >
                {error}
              </p>
              <button
                onClick={fetchEmployees}
                className={`px-4 py-2 rounded-lg font-medium transition-colors ${
                  isLight
                    ? "bg-blue-500 hover:bg-blue-600 text-white"
                    : "bg-blue-600 hover:bg-blue-700 text-white"
                }`}
              >
                Try Again
              </button>
            </div>
          )}

          {!loading && !error && employees.length === 0 && (
            <div
              className={`text-center py-8 ${
                isLight ? "text-gray-600" : "text-gray-400"
              }`}
            >
              <FaUsers className="text-4xl mx-auto mb-3 text-gray-400" />
              <h3 className="text-lg font-semibold mb-2">No Employees Found</h3>
              <p className="text-sm">
                No employees are currently assigned to this department.
              </p>
            </div>
          )}

          {!loading && !error && employees.length > 0 && (
            <div className={`space-y-3 ${employees.length > 3 ? "pr-2" : ""}`}>
              {employees.map((employee, index) => (
                <div
                  key={employee.employee_id || index}
                  className={`p-4 rounded-lg border transition-colors ${
                    isLight
                      ? "bg-gray-50 border-gray-200 hover:bg-yellow-50 hover:border-yellow-200"
                      : "bg-gray-700 border-gray-600 hover:bg-blue-900 hover:border-blue-600"
                  }`}
                >
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-3">
                      <div
                        className={`p-2 rounded-full ${
                          isLight ? "bg-blue-100" : "bg-blue-900"
                        }`}
                      >
                        <FaUserTie className={`text-blue-600 text-sm`} />
                      </div>
                      <div>
                        <h4
                          className={`font-semibold ${
                            isLight ? "text-gray-900" : "text-white"
                          }`}
                        >
                          {employee.employee_name}
                        </h4>
                        <div className="flex items-center gap-4 mt-1">
                          <div className="flex items-center gap-1">
                            <FaBuilding
                              className={`text-xs ${
                                isLight ? "text-gray-500" : "text-gray-400"
                              }`}
                            />
                            <span
                              className={`text-xs ${
                                isLight ? "text-gray-600" : "text-gray-400"
                              }`}
                            >
                              {employee.division_name}
                            </span>
                          </div>
                          <span
                            className={`text-xs px-2 py-1 rounded-full ${
                              isLight
                                ? "bg-green-100 text-green-700"
                                : "bg-green-900 text-green-300"
                            }`}
                          >
                            {employee.role_name || "No Role"}
                          </span>
                        </div>
                      </div>
                    </div>
                    {employee.remarks && (
                      <div
                        className={`text-xs text-right ${
                          isLight ? "text-gray-500" : "text-gray-400"
                        }`}
                      >
                        <p className="max-w-xs truncate">{employee.remarks}</p>
                      </div>
                    )}
                  </div>
                </div>
              ))}
            </div>
          )}
        </div>
      </div>
    </div>
  );
};

export default DepartmentEmployeesModal;
