import React, { useState, useEffect } from "react";
import { useTheme } from "../../contexts/ThemeContext";
import {
  FaExclamationTriangle,
  FaSpinner,
  FaExclamationCircle,
  FaTimesCircle,
  FaChevronLeft,
  FaChevronRight,
} from "react-icons/fa";
import { API_BASE_URL_WITH_API } from "../../utils/config";

const OverdueProjects = () => {
  const { isLight } = useTheme();
  const [projects, setProjects] = useState([]);
  const [loading, setLoading] = useState(false);
  const [error, setError] = useState(null);
  const [currentPage, setCurrentPage] = useState(1);
  const [itemsPerPage] = useState(10);

  useEffect(() => {
    fetchOverdueProjects();
  }, []);

  const fetchOverdueProjects = async () => {
    setLoading(true);
    setError(null);

    try {
      const response = await fetch(
        `${API_BASE_URL_WITH_API}/filterations/overdue_projects`
      );
      if (!response.ok) {
        throw new Error(`HTTP error! status: ${response.status}`);
      }

      const data = await response.json();
      setProjects(data);
    } catch (err) {
      console.error("Error fetching overdue projects:", err);
      setError("Failed to load overdue projects data");
    } finally {
      setLoading(false);
    }
  };

  const getSeverityColor = (severityLevel) => {
    switch (severityLevel) {
      case "critical":
        return isLight ? "text-red-600" : "text-red-400";
      case "high":
        return isLight ? "text-orange-600" : "text-orange-400";
      case "medium":
        return isLight ? "text-yellow-600" : "text-yellow-400";
      default:
        return isLight ? "text-amber-600" : "text-amber-400";
    }
  };

  const getSeverityIcon = (severityLevel) => {
    switch (severityLevel) {
      case "critical":
        return <FaTimesCircle className="text-red-500 flex-shrink-0" />;
      case "high":
        return <FaExclamationCircle className="text-orange-500 flex-shrink-0" />;
      case "medium":
        return <FaExclamationTriangle className="text-yellow-500 flex-shrink-0" />;
      default:
        return <FaExclamationTriangle className="text-amber-500 flex-shrink-0" />;
    }
  };

  const getDaysOverdueText = (daysOverdue) => {
    if (daysOverdue === 1) return "1 Day Overdue";
    return `${daysOverdue} Days Overdue`;
  };

  // Pagination logic
  const indexOfLastItem = currentPage * itemsPerPage;
  const indexOfFirstItem = indexOfLastItem - itemsPerPage;
  const currentProjects = projects.slice(indexOfFirstItem, indexOfLastItem);
  const totalPages = Math.ceil(projects.length / itemsPerPage);

  const goToPage = (page) => {
    setCurrentPage(page);
  };

  const goToPrevious = () => {
    setCurrentPage((prev) => Math.max(prev - 1, 1));
  };

  const goToNext = () => {
    setCurrentPage((prev) => Math.min(prev + 1, totalPages));
  };

  const cardBgClass = isLight
    ? "bg-white border border-yellow-200 shadow-md hover:shadow-lg"
    : "bg-slate-800 border border-slate-700 shadow-xl hover:shadow-2xl";

  const headingClass = isLight ? "text-amber-800" : "text-slate-100";

  const textClass = isLight ? "text-amber-700" : "text-slate-300";

  if (error) {
    return (
      <div
        className={`${cardBgClass} rounded-xl overflow-hidden transition-all duration-300`}
      >
        <div className="p-4 sm:p-8 text-center">
          <div className={`${textClass}`}>
            <div className="text-red-500 text-3xl sm:text-4xl mb-4">⚠️</div>
            <p className="text-base sm:text-lg font-medium mb-2 text-red-500">
              Error Loading Data
            </p>
            <p className="text-sm opacity-75">{error}</p>
            <button
              onClick={fetchOverdueProjects}
              className="mt-4 px-3 sm:px-4 py-2 bg-gradient-to-r from-yellow-500 to-amber-500 hover:from-yellow-600 hover:to-amber-600 text-white rounded-lg transition-all duration-200 text-sm"
            >
              Try Again
            </button>
          </div>
        </div>
      </div>
    );
  }

  return (
    <div
      className={`${cardBgClass} rounded-xl overflow-hidden transition-all duration-300`}
    >
      {/* Header */}
      <div className={`px-4 sm:px-6 py-3 sm:py-4 border-b ${
        isLight ? "border-gray-200" : "border-slate-600"
      }`}>
        <h2
          className={`text-lg sm:text-2xl font-bold ${headingClass} flex items-center gap-2`}
        >
          <FaExclamationTriangle
            className={`${isLight ? "text-amber-600" : "text-gray-400"} text-lg sm:text-xl flex-shrink-0`}
          />
          Overdue Projects
        </h2>
      </div>

      {loading ? (
        <div className="p-4 sm:p-8 text-center">
          <div className={`${textClass}`}>
            <div className="animate-spin h-6 w-6 sm:h-8 sm:w-8 border-4 border-blue-600 border-t-transparent rounded-full mx-auto mb-4"></div>
            <p className="text-base sm:text-lg font-medium">Loading overdue projects...</p>
          </div>
        </div>
      ) : projects.length === 0 ? (
        <div className="p-4 sm:p-8 text-center">
          <div className={`${textClass}`}>
            <div className="text-3xl sm:text-4xl mb-4">📋</div>
            <p className="text-base sm:text-lg font-medium mb-2">No Overdue Projects</p>
            <p className="text-sm opacity-75">
              All projects are on track with their deadlines.
            </p>
          </div>
        </div>
      ) : (
        <div className="overflow-x-auto">
          <table className="w-full table-fixed">
            <thead
              className={`${
                isLight
                  ? "bg-gradient-to-r from-yellow-200 to-amber-200"
                  : "bg-slate-700"
              }`}
            >
              <tr>
                <th
                  className={`px-2 sm:px-3 py-2 text-center font-semibold text-xs ${headingClass}`}
                  style={{ width: "30%" }}
                >
                  Project Name
                </th>
                <th
                  className={`px-2 sm:px-3 py-2 text-center font-semibold text-xs ${headingClass}`}
                  style={{ width: "20%" }}
                >
                  Priority
                </th>
                <th
                  className={`px-2 sm:px-3 py-2 text-center font-semibold text-xs ${headingClass}`}
                  style={{ width: "25%" }}
                >
                  Project Lead
                </th>
                <th
                  className={`px-2 sm:px-3 py-2 text-center font-semibold text-xs ${headingClass}`}
                  style={{ width: "25%" }}
                >
                  Delayed (days)
                </th>
              </tr>
            </thead>
            <tbody className={isLight ? "bg-white" : "bg-gray-800"}>
              {currentProjects.map((project, index) => (
                <tr
                  key={project.project_id}
                  className={`border-b transition-colors duration-200 ${
                    isLight
                      ? "border-gray-200 hover:bg-yellow-50"
                      : "border-slate-600 hover:bg-slate-700"
                  } ${
                    index % 2 === 0
                      ? isLight
                        ? "bg-gray-50"
                        : "bg-gray-750"
                      : isLight
                      ? "bg-white"
                      : "bg-gray-800"
                  }`}
                >
                  <td
                    className={`px-1 sm:px-2 py-2 text-center font-medium text-xs truncate ${textClass}`}
                    title={project.project_name}
                  >
                    {project.project_name}
                  </td>
                  <td className="px-1 sm:px-2 py-2 text-center">
                    <span
                      className={`inline-flex items-center gap-1 px-1.5 sm:px-2 py-1 rounded-full text-xs font-medium ${
                        project.project_priority === "High"
                          ? isLight
                            ? "bg-red-100 text-red-800"
                            : "bg-red-900 text-red-200"
                          : project.project_priority === "Medium"
                          ? isLight
                            ? "bg-orange-100 text-orange-800"
                            : "bg-orange-900 text-orange-200"
                          : isLight
                          ? "bg-green-100 text-green-800"
                          : "bg-green-900 text-green-200"
                      }`}
                    >
                      {project.project_priority}
                    </span>
                  </td>
                  <td
                    className={`px-1 sm:px-2 py-2 text-center text-xs truncate ${textClass}`}
                    title={project.project_lead}
                  >
                    {project.project_lead}
                  </td>
                  <td className="px-1 sm:px-2 py-2 text-center">
                    <div className="flex items-center justify-center gap-1 sm:gap-2">
                      {getSeverityIcon(project.severity_level)}
                      <span
                        className={`font-semibold text-xs ${getSeverityColor(
                          project.severity_level
                        )}`}
                      >
                        {getDaysOverdueText(project.days_overdue)}
                      </span>
                    </div>
                  </td>
                </tr>
              ))}
            </tbody>
          </table>
        </div>
      )}

      {/* Pagination Controls */}
      {totalPages > 1 && (
        <div
          className={`mt-0 px-3 py-2 border-t overflow-hidden ${
            isLight
              ? "bg-gray-50 border-gray-200"
              : "bg-slate-700 border-slate-600"
          }`}
        >
          <div className="flex flex-col sm:flex-row justify-between items-center gap-3 sm:gap-2 min-w-0">
            {/* Page Info */}
            <div
              className={`text-xs flex-shrink-0 text-center sm:text-left ${
                isLight ? "text-gray-600" : "text-gray-300"
              }`}
            >
              {indexOfFirstItem + 1}-
              {Math.min(indexOfLastItem, projects.length)} of {projects.length}
            </div>

            {/* Pagination Controls */}
            <div className="flex items-center justify-center gap-2">
              {/* Previous Button */}
              <button
                onClick={goToPrevious}
                disabled={currentPage === 1}
                className={`flex items-center justify-center gap-1 px-3 py-2 rounded-lg text-sm ${
                  currentPage === 1
                    ? (isLight ? 'bg-gray-100 text-gray-400 cursor-not-allowed' : 'bg-gray-600 text-gray-500 cursor-not-allowed')
                    : (isLight ? 'bg-white hover:bg-gray-100 text-gray-700 shadow-sm' : 'bg-gray-600 hover:bg-gray-500 text-gray-200')
                }`}
              >
                <FaChevronLeft className="text-xs flex-shrink-0" />
                <span>Prev</span>
              </button>

              {/* Current Page */}
              <div className={`px-3 py-2 rounded-lg font-medium text-sm ${
                isLight ? 'bg-amber-500 text-white' : 'bg-gray-300 text-gray-900'
              }`}>
                {currentPage}
              </div>

              {/* Next Button */}
              <button
                onClick={goToNext}
                disabled={currentPage === totalPages}
                className={`flex items-center justify-center gap-1 px-3 py-2 rounded-lg text-sm ${
                  currentPage === totalPages
                    ? (isLight ? 'bg-gray-100 text-gray-400 cursor-not-allowed' : 'bg-gray-600 text-gray-500 cursor-not-allowed')
                    : (isLight ? 'bg-white hover:bg-gray-100 text-gray-700 shadow-sm' : 'bg-gray-600 hover:bg-gray-500 text-gray-200')
                }`}
              >
                <span>Next</span>
                <FaChevronRight className="text-xs flex-shrink-0" />
              </button>
            </div>
          </div>
        </div>
      )}

      {projects.length > 0 && (
        <div
          className={`mx-4 sm:mx-6 mt-3 sm:mt-4 px-4 sm:px-6 py-2 sm:py-3 rounded-lg text-xs sm:text-sm ${
            isLight
              ? "bg-amber-50 border border-amber-200 text-amber-700"
              : "bg-slate-700 border border-slate-600 text-gray-300"
          }`}
        >
          <div className="flex items-center gap-2">
            <FaExclamationTriangle className="text-amber-500 flex-shrink-0" />
            <span>
              Showing {projects.length} overdue project
              {projects.length !== 1 ? "s" : ""}
              (sorted by most delayed first)
            </span>
          </div>
        </div>
      )}
      
      {/* Bottom padding for consistent spacing */}
      <div className="pb-4 sm:pb-6"></div>
    </div>
  );
};

export default OverdueProjects;
