import React, { useState, useEffect, useMemo, useCallback } from "react";
import Navbar from "./Navbar";
import Sidebar from "./Sidebar";
import { useTheme } from "../contexts/ThemeContext";
import DashboardStatus from "./Dashboard/DashboardStatus";

// Import all page components
import DashboardHome from "../pages/DashboardHome";
import EmployeesPage from "../pages/EmployeesPage";
import EmployeeLogPage from "../pages/EmployeeLogPage";
import ProjectsPage from "../pages/ProjectsPage";
import ProjectLogPage from "../pages/ProjectLogPage";
import FormsPage from "../pages/FormsPage";
import ProjectLeadPage from "../pages/ProjectLeadPage";
import DepartmentsPage from "../pages/DepartmentsPage";

const Dashboard = () => {
  // Persist active section in localStorage to maintain state after page refresh
  const [activeSection, setActiveSection] = useState(() => {
    const savedSection = localStorage.getItem("activeSection");
    return savedSection || "dashboard";
  });
  const [sidebarExpanded, setSidebarExpanded] = useState(false);
  const [isMobileMenuOpen, setIsMobileMenuOpen] = useState(false);
  const [windowWidth, setWindowWidth] = useState(
    typeof window !== "undefined" ? window.innerWidth : 1024
  );
  const { isLight } = useTheme();

  // Handle window resize for responsive behavior
  useEffect(() => {
    const handleResize = () => {
      setWindowWidth(window.innerWidth);
      // Auto-close mobile menu on desktop
      if (window.innerWidth >= 1024 && isMobileMenuOpen) {
        setIsMobileMenuOpen(false);
      }
    };

    window.addEventListener("resize", handleResize);
    return () => window.removeEventListener("resize", handleResize);
  }, [isMobileMenuOpen]);

  // Close mobile menu when clicking outside or navigating
  useEffect(() => {
    const handleEscape = (e) => {
      if (e.key === "Escape" && isMobileMenuOpen) {
        setIsMobileMenuOpen(false);
      }
    };

    document.addEventListener("keydown", handleEscape);
    return () => document.removeEventListener("keydown", handleEscape);
  }, [isMobileMenuOpen]);

  const handleMobileMenuToggle = useCallback(() => {
    setIsMobileMenuOpen(!isMobileMenuOpen);
  }, [isMobileMenuOpen]);

  const handleMobileMenuClose = useCallback(() => {
    setIsMobileMenuOpen(false);
  }, []);

  const handleSidebarToggle = useCallback((expanded) => {
    setSidebarExpanded(expanded);
  }, []);

  const handleSectionChange = useCallback(
    (section) => {
      setActiveSection(section);
      // Save active section to localStorage for persistence
      localStorage.setItem("activeSection", section);
      // Auto-close mobile menu when section changes on mobile
      if (windowWidth < 1024) {
        setIsMobileMenuOpen(false);
      }
    },
    [windowWidth, isMobileMenuOpen]
  );

  // Dynamic layout calculations
  const isDesktop = windowWidth >= 1024;
  const isTablet = windowWidth >= 768 && windowWidth < 1024;
  const isMobile = windowWidth < 768;

  // Calculate main content margin based on screen size and sidebar state
  const getMainContentClasses = useMemo(() => {
    let classes = "transition-all duration-300 pt-20"; // Increased from pt-16 to pt-20 for more top space

    if (isDesktop) {
      // Desktop: Always show sidebar, expand on hover
      if (sidebarExpanded) {
        classes += " ml-48"; // Expanded sidebar width
      } else {
        classes += " ml-12"; // Collapsed sidebar width
      }
    } else {
      // Mobile/Tablet: No permanent margin, overlay sidebar
      classes += " ml-0";
    }

    // Add responsive padding with reduced side spacing and extra top padding
    if (isMobile) {
      classes += " px-2 py-3 pt-6"; // Reduced side padding on mobile
    } else if (isTablet) {
      classes += " px-3 py-4 pt-8"; // Reduced side padding on tablet
    } else {
      classes += " px-4 py-6 pt-10"; // Reduced side padding on desktop
    }

    return classes;
  }, [isDesktop, isTablet, isMobile, sidebarExpanded]);

  // Enhanced theme-based styling with beautiful colors
  const backgroundClass = isLight
    ? "bg-gradient-to-br from-yellow-50 via-amber-50 to-orange-50"
    : "bg-slate-900";

  // Render the appropriate page component based on active section
  const renderCurrentPage = useMemo(() => {
    // Wrap content with reduced container spacing
    const PageWrapper = ({ children }) => (
      <div className="max-w-full mx-auto">{children}</div>
    );

    switch (activeSection) {
      case "dashboard":
        return (
          <PageWrapper>
            <DashboardHome onSectionChange={setActiveSection} />
          </PageWrapper>
        );
      case "employees":
        return (
          <PageWrapper>
            <EmployeesPage onSectionChange={setActiveSection} />
          </PageWrapper>
        );

      case "employee_log":
        return (
          <PageWrapper>
            <EmployeeLogPage />
          </PageWrapper>
        );
      case "projects":
        return (
          <PageWrapper>
            <ProjectsPage />
          </PageWrapper>
        );
      case "project_log":
        return (
          <PageWrapper>
            <ProjectLogPage />
          </PageWrapper>
        );
      case "forms":
        return (
          <PageWrapper>
            <FormsPage />
          </PageWrapper>
        );
      case "project_lead":
        return (
          <PageWrapper>
            <ProjectLeadPage onSectionChange={setActiveSection} />
          </PageWrapper>
        );
      case "departments":
        return (
          <PageWrapper>
            <DepartmentsPage />
          </PageWrapper>
        );
      default:
        return (
          <PageWrapper>
            <DashboardHome />
          </PageWrapper>
        );
    }
  }, [activeSection]);

  return (
    <div
      className={`relative ${backgroundClass} min-h-screen transition-all duration-300`}
    >
      {/* Overlay for mobile menu - positioned to cover content when right sidebar is open */}
      {isMobileMenuOpen && (
        <div
          className="fixed inset-0 bg-transparent backdrop-blur-[1px] z-30 lg:hidden"
          onClick={handleMobileMenuClose}
        />
      )}

      <Navbar
        onMobileMenuToggle={handleMobileMenuToggle}
        isMobileMenuOpen={isMobileMenuOpen}
      />

      <Sidebar
        activeSection={activeSection}
        onSectionChange={handleSectionChange}
        onSidebarToggle={handleSidebarToggle}
        isMobileMenuOpen={isMobileMenuOpen}
        onMobileMenuClose={handleMobileMenuClose}
      />

      <div className={getMainContentClasses}>
        {renderCurrentPage}
        {activeSection === "dashboard" && <DashboardStatus />}
      </div>
    </div>
  );
};

export default Dashboard;
