import React from "react";
import { useTheme } from "../../contexts/ThemeContext";
import {
  FaChevronLeft,
  FaChevronRight,
  FaProjectDiagram,
} from "react-icons/fa";
import ProjectActions from "./ProjectActions";
import { formatIST } from "../../utils/timeUtils";

const ProjectsTable = ({
  filteredProjects,
  currentProjects,
  onView,
  currentPage,
  totalPages,
  goToPage,
  goToPrevious,
  goToNext,
}) => {
  const { isLight } = useTheme();

  const cardBgClass = isLight
    ? "bg-white border border-yellow-200 shadow-md hover:shadow-lg"
    : "bg-gradient-to-br from-gray-800 to-slate-800 border border-gray-700 shadow-lg hover:shadow-xl";

  const headingClass = isLight ? "text-amber-800" : "text-gray-100";

  const textClass = isLight ? "text-amber-700" : "text-gray-300";

  const buttonClass = isLight
    ? "bg-gradient-to-r from-yellow-500 to-amber-500 hover:from-yellow-600 hover:to-amber-600 text-white"
    : "bg-gradient-to-r from-gray-600 to-slate-600 hover:from-gray-500 hover:to-slate-500 text-white";

  // Get status color classes
  const getStatusColor = (status) => {
    switch (status?.toLowerCase()) {
      case "completed":
        return isLight ? "text-green-700" : "text-green-400";
      case "in progress":
      case "ongoing":
        return isLight ? "text-blue-700" : "text-blue-400";
      case "planning":
      case "planned":
        return isLight ? "text-yellow-700" : "text-yellow-400";
      case "on hold":
      case "paused":
        return isLight ? "text-red-700" : "text-red-400";
      default:
        return isLight ? "text-gray-800" : "text-gray-200";
    }
  };

  // Get priority color classes
  const getPriorityColor = (priority) => {
    switch (priority?.toLowerCase()) {
      case "high":
        return isLight ? "text-red-700" : "text-red-400";
      case "medium":
        return isLight ? "text-yellow-700" : "text-yellow-400";
      case "low":
        return isLight ? "text-green-700" : "text-green-400";
      default:
        return isLight ? "text-gray-800" : "text-gray-200";
    }
  };

  return (
    <div
      className={`hidden md:block ${cardBgClass} rounded-xl overflow-hidden transition-all duration-300`}
    >
      <div className="overflow-hidden">
        <table className="w-full table-fixed">
          <thead
            className={`${
              isLight
                ? "bg-gradient-to-r from-yellow-200 to-amber-200"
                : "bg-gradient-to-r from-gray-700 to-slate-700"
            }`}
          >
            <tr>
              <th
                className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`}
                style={{ width: "18%" }}
              >
                Project
              </th>
              <th
                className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`}
                style={{ width: "14%" }}
              >
                Project Lead
              </th>
              <th
                className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`}
                style={{ width: "11%" }}
              >
                Type
              </th>
              <th
                className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`}
                style={{ width: "11%" }}
              >
                Status
              </th>
              <th
                className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`}
                style={{ width: "15%" }}
              >
                PO Status
              </th>
              <th
                className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`}
                style={{ width: "11%" }}
              >
                Deadline
              </th>
              <th
                className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`}
                style={{ width: "11%" }}
              >
                Progress
              </th>
              <th
                className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`}
                style={{ width: "11%" }}
              >
                Hours (A/E)
              </th>
              <th
                className={`px-2 py-2 text-center font-semibold text-xs ${headingClass}`}
                style={{ width: "8%" }}
              >
                Actions
              </th>
            </tr>
          </thead>
          <tbody className={isLight ? "bg-white" : "bg-gray-800"}>
            {filteredProjects.length === 0 ? (
              <tr>
                <td
                  colSpan="8"
                  className={`p-12 text-center ${
                    isLight ? "bg-white" : "bg-gray-800"
                  }`}
                >
                  <div className="flex flex-col items-center gap-3">
                    <FaProjectDiagram
                      className={`text-4xl ${
                        isLight ? "text-gray-300" : "text-gray-500"
                      }`}
                    />
                    <div className={`${textClass} text-lg`}>
                      No projects found.
                    </div>
                  </div>
                </td>
              </tr>
            ) : (
              currentProjects.map((project, index) => (
                <tr
                  key={project.project_id}
                  className={`border-b transition-colors duration-200 ${
                    isLight
                      ? "border-gray-200 hover:bg-yellow-50"
                      : "border-gray-700 hover:bg-gray-750"
                  } ${
                    index % 2 === 0
                      ? isLight
                        ? "bg-gray-50"
                        : "bg-gray-750"
                      : isLight
                      ? "bg-white"
                      : "bg-gray-800"
                  }`}
                >
                  <td className="px-2 py-2 text-center">
                    <div>
                      <div
                        className={`font-medium text-xs truncate ${textClass}`}
                        title={project.project_name}
                      >
                        {project.project_name}
                      </div>
                      <div
                        className={`text-xs ${
                          isLight ? "text-gray-600" : "text-gray-400"
                        }`}
                      >
                        ID: {project.project_id}
                      </div>
                    </div>
                  </td>
                  <td className="px-2 py-2 text-center">
                    <div
                      className={`text-xs truncate ${textClass}`}
                      title={project.project_lead_name || "Not Assigned"}
                    >
                      {project.project_lead_name || "Not Assigned"}
                    </div>
                  </td>
                  <td className="px-2 py-2 text-center">
                    <div
                      className={`text-xs truncate ${textClass}`}
                      title={project.type_name || "Not Assigned"}
                    >
                      {project.type_name || "Not Assigned"}
                    </div>
                  </td>
                  <td className="px-2 py-2 text-center">
                    <span
                      className={`px-2 py-1 rounded-full text-xs font-medium ${
                        project.status_name === "Completed"
                          ? "bg-green-100 text-green-800"
                          : project.status_name === "Ongoing"
                          ? "bg-blue-100 text-blue-800"
                          : project.status_name === "PO Received"
                          ? "bg-yellow-100 text-yellow-800"
                          : project.status_name === "Opened"
                          ? "bg-purple-100 text-purple-800"
                          : project.status_name === "Extended"
                          ? "bg-red-100 text-red-800"
                          : project.status_name === "Shelved"
                          ? "bg-gray-100 text-gray-600"
                          : "bg-gray-100 text-gray-800"
                      }`}
                    >
                      {project.status_name || "Not Assigned"}
                    </span>
                  </td>
                  <td className="px-2 py-2 text-center">
                    <span
                      className={`inline-block px-3 py-1 rounded-full text-xs font-medium whitespace-nowrap ${
                        project.po_status === "PO Received"
                          ? "bg-green-100 text-green-800"
                          : project.po_status === "PO Not Received"
                          ? "bg-yellow-100 text-yellow-800"
                          : "bg-gray-100 text-gray-800"
                      }`}
                    >
                      {project.po_status || "Not Set"}
                    </span>
                  </td>
                  <td className="px-2 py-2 text-center">
                    <div className={`text-xs ${textClass}`}>
                      {project.deadline ? formatIST(project.deadline) : "N/A"}
                    </div>
                  </td>
                  <td className="px-2 py-2 text-center">
                    <div className="w-12 bg-gray-200 rounded-full h-1.5 mb-1 mx-auto">
                      <div
                        className={`h-1.5 rounded-full ${
                          (project.percent_completed || 0) >= 80
                            ? "bg-green-600"
                            : (project.percent_completed || 0) >= 50
                            ? "bg-blue-600"
                            : (project.percent_completed || 0) >= 25
                            ? "bg-yellow-600"
                            : "bg-red-600"
                        }`}
                        style={{ width: `${project.percent_completed || 0}%` }}
                      ></div>
                    </div>
                    <span className="text-xs">
                      {project.percent_completed || 0}%
                    </span>
                  </td>
                  <td className="px-2 py-2 text-center">
                    <div className="flex flex-col items-center">
                      <span
                        className={`text-xs ${
                          isLight ? "text-gray-600" : "text-gray-400"
                        }`}
                      >
                        Est:{" "}
                        {project.estimated_hours
                          ? `${parseFloat(project.estimated_hours).toFixed(1)}h`
                          : "N/A"}
                      </span>
                      <span className="font-medium text-xs">
                        Act:{" "}
                        {project.actual_hours
                          ? `${parseFloat(project.actual_hours).toFixed(1)}h`
                          : "0.0h"}
                      </span>
                    </div>
                  </td>
                  <td className="px-1 py-2 text-center">
                    <ProjectActions onView={() => onView(project)} />
                  </td>
                </tr>
              ))
            )}
          </tbody>
        </table>
      </div>

      {/* Pagination Controls */}
      {totalPages > 1 && (
        <div
          className={`p-4 border-t ${
            isLight ? "border-yellow-200" : "border-gray-700"
          }`}
        >
          <div className="flex flex-col sm:flex-row justify-between items-center gap-3">
            {/* Page Info */}
            <div
              className={`text-sm text-center ${
                isLight ? "text-gray-600" : "text-gray-300"
              }`}
            >
              Showing {currentProjects.length} of {filteredProjects.length}{" "}
              projects
            </div>

            {/* Pagination Controls */}
            <div className="flex items-center justify-center gap-2">
              {/* Previous Button */}
              <button
                onClick={goToPrevious}
                disabled={currentPage === 1}
                className={`flex items-center justify-center gap-1 px-3 py-2 rounded-lg text-sm ${
                  currentPage === 1
                    ? (isLight ? 'bg-gray-100 text-gray-400 cursor-not-allowed' : 'bg-gray-600 text-gray-500 cursor-not-allowed')
                    : (isLight ? 'bg-white hover:bg-gray-100 text-gray-700 shadow-sm' : 'bg-gray-600 hover:bg-gray-500 text-gray-200')
                }`}
              >
                <FaChevronLeft className="text-xs flex-shrink-0" />
                <span>Prev</span>
              </button>

              {/* Current Page */}
              <div className={`px-3 py-2 rounded-lg font-medium text-sm ${
                isLight ? 'bg-amber-500 text-white' : 'bg-gray-300 text-gray-900'
              }`}>
                {currentPage}
              </div>

              {/* Next Button */}
              <button
                onClick={goToNext}
                disabled={currentPage === totalPages}
                className={`flex items-center justify-center gap-1 px-3 py-2 rounded-lg text-sm ${
                  currentPage === totalPages
                    ? (isLight ? 'bg-gray-100 text-gray-400 cursor-not-allowed' : 'bg-gray-600 text-gray-500 cursor-not-allowed')
                    : (isLight ? 'bg-white hover:bg-gray-100 text-gray-700 shadow-sm' : 'bg-gray-600 hover:bg-gray-500 text-gray-200')
                }`}
              >
                <span>Next</span>
                <FaChevronRight className="text-xs flex-shrink-0" />
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default ProjectsTable;
