import React from 'react';
import { useTheme } from '../../contexts/ThemeContext';
import { FaChevronLeft, FaChevronRight } from 'react-icons/fa';

const ProjectLogTable = ({ 
  projectLogData, 
  currentPage, 
  totalPages, 
  startIndex, 
  endIndex, 
  goToPrevious, 
  goToNext 
}) => {
  const { isLight } = useTheme();

  const cardBgClass = isLight 
    ? 'bg-white border border-yellow-200 shadow-md hover:shadow-lg' 
    : 'bg-slate-800 border border-slate-700 shadow-xl hover:shadow-2xl';
  
  const headingClass = isLight 
    ? 'text-amber-800' 
    : 'text-slate-100';
  
  const textClass = isLight 
    ? 'text-amber-700' 
    : 'text-slate-300';

  return (
    <div className={`${cardBgClass} rounded-xl overflow-hidden transition-all duration-300`}>
      <div className="overflow-x-auto">
        <table className="w-full table-fixed min-w-[1000px]">
          <thead className={`${isLight ? 'bg-gradient-to-r from-yellow-200 to-amber-200' : 'bg-slate-700'}`}>
            <tr>
              <th className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`} style={{ width: '5%' }}>Log ID</th>
              <th className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`} style={{ width: '5%' }}>User ID</th>
              <th className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`} style={{ width: '10%' }}>Project</th>
              <th className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`} style={{ width: '7%' }}>Type</th>
              <th className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`} style={{ width: '7%' }}>Status</th>
              <th className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`} style={{ width: '10%' }}>Invoice Status</th>
              <th className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`} style={{ width: '8%' }}>PO Status</th>
              <th className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`} style={{ width: '7%' }}>Priority</th>
              <th className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`} style={{ width: '7%' }}>Deadline</th>
              <th className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`} style={{ width: '7%' }}>Progress</th>
              <th className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`} style={{ width: '7%' }}>Hours</th>
              <th className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`} style={{ width: '7%' }}>Action</th>
              <th className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`} style={{ width: '8%' }}>Timestamp</th>
              <th className={`px-3 py-2 text-center font-semibold text-xs ${headingClass}`} style={{ width: '7%' }}>Remarks</th>
            </tr>
          </thead>
          <tbody className={isLight ? 'bg-white' : 'bg-gray-800'}>
            {projectLogData && projectLogData.length > 0 ? (
              projectLogData.map((log, index) => (
                <tr 
                  key={log.id} 
                  className={`border-b transition-colors duration-200 ${
                    isLight ? 'border-gray-200 hover:bg-yellow-50' : 'border-slate-600 hover:bg-slate-700'
                  } ${
                    index % 2 === 0 
                      ? (isLight ? 'bg-gray-50' : 'bg-gray-750') 
                      : (isLight ? 'bg-white' : 'bg-gray-800')
                  }`}
                >
                  <td className={`px-1 sm:px-2 py-2 text-center font-mono text-xs font-medium ${isLight ? 'text-blue-600' : 'text-cyan-400'}`}>
                    #{log.id}
                  </td>
                  <td className={`px-1 sm:px-2 py-2 text-center font-mono text-xs ${isLight ? 'text-gray-600' : 'text-slate-400'}`}>{log.user_id}</td>
                  <td className={`px-1 sm:px-2 py-2 text-center font-medium text-xs truncate ${textClass}`} title={log.project_name}>{log.project_name}</td>
                  <td className={`px-1 sm:px-2 py-2 text-center text-xs truncate ${textClass}`} title={log.type_name || 'N/A'}>{log.type_name || 'N/A'}</td>
                  <td className={`px-1 sm:px-2 py-2 text-center text-xs`}>
                    <div className="flex items-center justify-center">
                      <span className={`px-2 py-1 rounded-full text-xs font-medium ${
                        log.status_name 
                          ? isLight 
                            ? 'bg-green-100 text-green-800' 
                            : 'bg-green-900/30 text-green-400'
                          : isLight 
                            ? 'bg-gray-100 text-gray-600' 
                            : 'bg-slate-700 text-slate-400'
                      }`}>
                        {log.status_name || 'N/A'}
                      </span>
                    </div>
                  </td>
                  <td className={`px-1 sm:px-2 py-2 text-center text-xs`}>
                    <div className="flex items-center justify-center">
                      <span className={`px-2 py-1 rounded-full text-xs font-medium ${
                        log.invoice_status_name === 'Payment Received'
                          ? isLight ? 'bg-green-100 text-green-800' : 'bg-green-900/30 text-green-400'
                          : log.invoice_status_name === 'Pending'
                          ? isLight ? 'bg-yellow-100 text-yellow-800' : 'bg-yellow-900/30 text-yellow-400'
                          : log.invoice_status_name === 'Submitted'
                          ? isLight ? 'bg-blue-100 text-blue-800' : 'bg-blue-900/30 text-blue-400'
                          : log.invoice_status_name === 'Internal'
                          ? isLight ? 'bg-purple-100 text-purple-800' : 'bg-purple-900/30 text-purple-400'
                          : log.invoice_status_name === 'Not Submitted'
                          ? isLight ? 'bg-red-100 text-red-800' : 'bg-red-900/30 text-red-400'
                          : isLight ? 'bg-gray-100 text-gray-600' : 'bg-slate-700 text-slate-400'
                      }`}>
                        {log.invoice_status_name || 'N/A'}
                      </span>
                    </div>
                  </td>
                  <td className={`px-1 sm:px-2 py-2 text-center text-xs`}>
                    <div className="flex items-center justify-center">
                      <span className={`px-2 py-1 rounded-full text-xs font-medium ${
                        log.po_status_name === 'PO Received'
                          ? isLight ? 'bg-green-100 text-green-800' : 'bg-green-900/30 text-green-400'
                          : log.po_status_name === 'PO Not Received'
                          ? isLight ? 'bg-red-100 text-red-800' : 'bg-red-900/30 text-red-400'
                          : isLight ? 'bg-gray-100 text-gray-600' : 'bg-slate-700 text-slate-400'
                      }`}>
                        {log.po_status_name || 'N/A'}
                      </span>
                    </div>
                  </td>
                  <td className={`px-1 sm:px-2 py-2 text-center text-xs`}>
                    <div className="flex items-center justify-center">
                      <span className={`px-2 py-1 rounded-full text-xs font-medium ${
                        log.priority_level_name === 'High'
                          ? isLight ? 'bg-red-100 text-red-800' : 'bg-red-900/30 text-red-400'
                          : log.priority_level_name === 'Medium'
                          ? isLight ? 'bg-yellow-100 text-yellow-800' : 'bg-yellow-900/30 text-yellow-400'
                          : log.priority_level_name === 'Low'
                          ? isLight ? 'bg-blue-100 text-blue-800' : 'bg-blue-900/30 text-blue-400'
                          : isLight ? 'bg-gray-100 text-gray-600' : 'bg-slate-700 text-slate-400'
                      }`}>
                        {log.priority_level_name || 'N/A'}
                      </span>
                    </div>
                  </td>
                  <td className={`px-1 sm:px-2 py-2 text-center text-xs ${textClass}`}>
                    <div className="flex items-center justify-center gap-1">
                      {log.deadline ? (
                        <>
                          <span className={`text-xs ${
                            new Date(log.deadline) < new Date() 
                              ? isLight ? 'text-red-600' : 'text-red-400'
                              : isLight ? 'text-gray-600' : 'text-slate-400'
                          }`}>
                            {log.deadline}
                          </span>
                          {new Date(log.deadline) < new Date() && (
                            <span className="text-xs text-red-500">⚠️</span>
                          )}
                        </>
                      ) : (
                        <span className={`text-xs ${isLight ? 'text-gray-400' : 'text-slate-500'}`}>
                          No deadline
                        </span>
                      )}
                    </div>
                  </td>
                  <td className={`px-1 sm:px-2 py-2 text-center text-xs ${textClass}`}>
                    <div className="flex items-center justify-center gap-2">
                      <div className={`w-10 h-1.5 rounded-full ${isLight ? 'bg-gray-200' : 'bg-slate-600'}`}>
                        <div 
                          className={`h-1.5 rounded-full ${isLight ? 'bg-blue-500' : 'bg-cyan-400'}`}
                          style={{ width: `${Math.min(100, log.percent_completed || 0)}%` }}
                        ></div>
                      </div>
                      <span className="text-xs">{log.percent_completed || 0}%</span>
                    </div>
                  </td>
                  <td className={`px-1 sm:px-2 py-2 text-center text-xs ${textClass}`}>
                    <div className="space-y-1">
                      <div className="text-xs">Est: {log.estimated_hours || 0}h</div>
                      <div className="text-xs">Act: {log.actual_hours || 0}h</div>
                    </div>
                  </td>
                  <td className={`px-1 sm:px-2 py-2 text-center text-xs truncate ${textClass}`} title={log.action}>{log.action}</td>
                  <td className={`px-1 sm:px-2 py-2 text-center text-xs ${isLight ? 'text-gray-600' : 'text-slate-400'}`}>{log.timestamp}</td>
                  <td className={`px-1 sm:px-2 py-2 text-center text-xs ${isLight ? 'text-gray-600' : 'text-slate-400'}`}>
                    <div className="truncate" title={log.remarks}>
                      {log.remarks}
                    </div>
                  </td>
                </tr>
              ))
            ) : (
              <tr>
                <td colSpan="14" className="p-8 text-center">
                  <div className={`${textClass}`}>
                    <p className="text-base sm:text-lg font-medium mb-2">No project logs found</p>
                    <p className="text-sm opacity-75">Try adjusting your date range or check back later.</p>
                  </div>
                </td>
              </tr>
            )}
          </tbody>
        </table>
      </div>

      {/* Pagination Controls */}
      {totalPages > 1 && (
        <div className={`mt-0 px-3 py-2 border-t overflow-hidden ${
          isLight 
            ? 'bg-gray-50 border-gray-200' 
            : 'bg-slate-700 border-slate-600'
        }`}>
          <div className="flex flex-col sm:flex-row justify-between items-center gap-3 sm:gap-2 min-w-0">
            {/* Page Info */}
            <div className={`text-xs flex-shrink-0 text-center sm:text-left ${
              isLight ? 'text-gray-600' : 'text-gray-300'
            }`}>
              {startIndex + 1}-{Math.min(endIndex, projectLogData.length)} of {projectLogData.length}
            </div>

            {/* Pagination Controls */}
            <div className="flex items-center justify-center gap-2">
              {/* Previous Button */}
              <button
                onClick={goToPrevious}
                disabled={currentPage === 1}
                className={`flex items-center justify-center gap-1 px-3 py-2 rounded-lg text-sm ${
                  currentPage === 1
                    ? (isLight ? 'bg-gray-100 text-gray-400 cursor-not-allowed' : 'bg-gray-600 text-gray-500 cursor-not-allowed')
                    : (isLight ? 'bg-white hover:bg-gray-100 text-gray-700 shadow-sm' : 'bg-gray-600 hover:bg-gray-500 text-gray-200')
                }`}
              >
                <FaChevronLeft className="text-xs flex-shrink-0" />
                <span>Prev</span>
              </button>

              {/* Current Page */}
              <div className={`px-3 py-2 rounded-lg font-medium text-sm ${
                isLight ? 'bg-amber-500 text-white' : 'bg-gray-300 text-gray-900'
              }`}>
                {currentPage}
              </div>

              {/* Next Button */}
              <button
                onClick={goToNext}
                disabled={currentPage === totalPages}
                className={`flex items-center justify-center gap-1 px-3 py-2 rounded-lg text-sm ${
                  currentPage === totalPages
                    ? (isLight ? 'bg-gray-100 text-gray-400 cursor-not-allowed' : 'bg-gray-600 text-gray-500 cursor-not-allowed')
                    : (isLight ? 'bg-white hover:bg-gray-100 text-gray-700 shadow-sm' : 'bg-gray-600 hover:bg-gray-500 text-gray-200')
                }`}
              >
                <span>Next</span>
                <FaChevronRight className="text-xs flex-shrink-0" />
              </button>
            </div>
          </div>
        </div>
      )}
    </div>
  );
};

export default ProjectLogTable; 